# pip install PySDL3
# py main.py

import ctypes
import os
import time

os.environ["SDL_MAIN_USE_CALLBACKS"] = "1"
os.environ["SDL_RENDER_DRIVER"] = "opengl"

import sdl3

renderer = ctypes.POINTER(sdl3.SDL_Renderer)()
window = ctypes.POINTER(sdl3.SDL_Window)()
spriteTexture = None

frameIndex = 0
lastTime = 0
animationTime = 0
animationInterval = 0.1

# Define animation frame
frames = []
frames.append(sdl3.SDL_FRect(0, 0, 96, 96)) # Frame 0
frames.append(sdl3.SDL_FRect(96, 0, 96, 96)) # Frame 1
frames.append(sdl3.SDL_FRect(192, 0, 96, 96)) # Frame 2
frames.append(sdl3.SDL_FRect(0, 96, 96, 96)) # Frame 3
frames.append(sdl3.SDL_FRect(96, 96, 96, 96)) # Frame 4
frames.append(sdl3.SDL_FRect(192, 96, 96, 96)) # Frame 5

@sdl3.SDL_AppInit_func
def SDL_AppInit(appstate, argc, argv):
    global spriteTexture

    if not sdl3.SDL_Init(sdl3.SDL_INIT_VIDEO):
        sdl3.SDL_Log("Couldn't initialize SDL: %s".encode() % sdl3.SDL_GetError())
        return sdl3.SDL_APP_FAILURE

    if not sdl3.SDL_CreateWindowAndRenderer("Sprite animation using PySDL3".encode(),
            350, 200, 0, window, renderer):
        sdl3.SDL_Log("Couldn't create window/renderer: %s".encode() % sdl3.SDL_GetError())
        return sdl3.SDL_APP_FAILURE

    sdl3.SDL_SetRenderVSync(renderer, 1) # Turn on vertical sync

    # Get the compiled SDL version
    compiled = sdl3.SDL_VERSION
    compiled_major = sdl3.SDL_VERSIONNUM_MAJOR(compiled)
    compiled_minor = sdl3.SDL_VERSIONNUM_MINOR(compiled)
    compiled_micro = sdl3.SDL_VERSIONNUM_MICRO(compiled)
    print(f"Compiled SDL version: {compiled_major}.{compiled_minor}.{compiled_micro}")

    # Get the linked SDL version
    linked = sdl3.SDL_GetVersion()
    linked_major = sdl3.SDL_VERSIONNUM_MAJOR(linked)
    linked_minor = sdl3.SDL_VERSIONNUM_MINOR(linked)
    linked_micro = sdl3.SDL_VERSIONNUM_MICRO(linked)
    print(f"Linked SDL version:   {linked_major}.{linked_minor}.{linked_micro}")

    texturePath = "./assets/sprites/warrior/idle.png".encode()
    surface = sdl3.SDL_LoadPNG(texturePath)
    if not surface:
        sdl.SDL_Log("PNG load failed: %s: %s".encode() % (texturePath, sdl3.SDL_GetError()))
        return sdl3.SDL_APP_FAILURE

    spriteTexture = sdl3.SDL_CreateTextureFromSurface(renderer, surface)
    sdl3.SDL_DestroySurface(surface)

    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppEvent_func
def SDL_AppEvent(appstate, event):
    if sdl3.SDL_DEREFERENCE(event).type == sdl3.SDL_EVENT_QUIT:
        return sdl3.SDL_APP_SUCCESS

    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppIterate_func
def SDL_AppIterate(appstate):
    global animationTime
    global lastTime
    global frameIndex

    # As you can see from this, rendering draws over whatever was drawn before it
    sdl3.SDL_SetRenderDrawColor(renderer, 153, 207, 145, sdl3.SDL_ALPHA_OPAQUE)
    sdl3.SDL_RenderClear(renderer) # Start with a blank canvas

    # Update animation
    srcRect = frames[frameIndex]

    # Render the current frame
    destRect = sdl3.SDL_FRect(50, 50, 96, 96)
    sdl3.SDL_RenderTexture(renderer, spriteTexture, srcRect, destRect)

    sdl3.SDL_RenderPresent(renderer)

    currentTime = sdl3.SDL_GetTicks()
    deltaTime = (currentTime - lastTime) / 1000
    lastTime = currentTime

    animationTime += deltaTime
    if animationTime > animationInterval:
        frameIndex = (frameIndex + 1) % 6
        animationTime = 0

    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppQuit_func
def SDL_AppQuit(appstate, result):

    sdl3.SDL_DestroyTexture(spriteTexture)
