#define SDL_MAIN_USE_CALLBACKS 1 // Use the callbacks instead of main()
#include <SDL3/SDL.h>
#include <SDL3/SDL_main.h>
#include <stdio.h>

// We will use this renderer to draw into this window every frame
static SDL_Window *window = NULL;
static SDL_Renderer *renderer = NULL;
static SDL_Texture *spriteTexture = NULL;

int frameIndex = 0;
uint32_t lastTime = 0;
float animationTime = 0.f;
const float animationPeriod = 0.1f;

// Define animation frames
SDL_FRect frames[6] = {
    { 0, 0, 96, 96 },   // Frame 0
    { 96, 0, 96, 96 },  // Frame 1
    { 192, 0, 96, 96 }, // Frame 2
    { 0, 96, 96, 96 },  // Frame 3
    { 96, 96, 96, 96 }, // Frame 4
    { 192, 96, 96, 96 } // Frame 5
};

// This function runs once at startup
SDL_AppResult SDL_AppInit(void **appstate, int argc, char *argv[])
{
    if (!SDL_Init(SDL_INIT_VIDEO))
    {
        SDL_Log("Couldn't initialize SDL: %s", SDL_GetError());
        return SDL_APP_FAILURE;
    }

    const char *title = "Sprite animation (PNG / native SDL 3.3 / C)";
    if (!SDL_CreateWindowAndRenderer(title, 400, 400, 0, &window, &renderer))
    {
        SDL_Log("Couldn't create window/renderer: %s", SDL_GetError());
        return SDL_APP_FAILURE;
    }

    SDL_SetRenderVSync(renderer, 1);

    // --- Print SDL versions ---
    printf("Compiled SDL3 version: %d.%d.%d\n",
        SDL_MAJOR_VERSION,
        SDL_MINOR_VERSION,
        SDL_MICRO_VERSION);
    // Get the version of the SDL library linked at runtime
    int v = SDL_GetVersion();
    printf("Linked SDL3 version:   %d.%d.%d\n", SDL_VERSIONNUM_MAJOR(v),
        SDL_VERSIONNUM_MINOR(v), SDL_VERSIONNUM_MICRO(v));

    const char *texturePath = "./assets/sprites/warrior/idle.png";
    SDL_Surface *surface = SDL_LoadPNG(texturePath);
    if (!surface)
    {
        SDL_Log("PNG load failed: %s: %s\n", texturePath, SDL_GetError());
        return SDL_APP_FAILURE;
    }

    spriteTexture = SDL_CreateTextureFromSurface(renderer, surface);
    SDL_DestroySurface(surface);

    return SDL_APP_CONTINUE;
}

// This function runs when a new event (mouse input, keypresses, etc) occurs
SDL_AppResult SDL_AppEvent(void *appstate, SDL_Event *event)
{
    if (event->type == SDL_EVENT_QUIT)
    {
        return SDL_APP_SUCCESS; // End the program, reporting success to the OS
    }
    return SDL_APP_CONTINUE;
}

// This function runs once per frame, and is the heart of the program
SDL_AppResult SDL_AppIterate(void *appstate)
{
    // As you can see from this, rendering draws
    // over whatever was drawn before it
    SDL_SetRenderDrawColor(renderer, 153, 207, 145, 255);
    SDL_RenderClear(renderer); // Start with a blank canvas

    // Update animation
    SDL_FRect srcRect = frames[frameIndex];

    // Render the current animation frame
    SDL_FRect destRect = { 50, 50, 96, 96 };
    SDL_RenderTexture(renderer, spriteTexture, &srcRect, &destRect);

    SDL_RenderPresent(renderer);

    // Calculate delta time
    uint32_t currentTime = SDL_GetTicks();
    float deltaTime = (currentTime - lastTime) / 1000.0f;
    lastTime = currentTime;

    animationTime += deltaTime;

    if (animationTime > animationPeriod)
    {
        frameIndex = (frameIndex + 1) % 6;
        animationTime = 0.0f;
    }

    return SDL_APP_CONTINUE;
}

// This function runs once at shutdown
void SDL_AppQuit(void *appstate, SDL_AppResult result)
{
    if (spriteTexture)
    {
        SDL_DestroyTexture(spriteTexture);
        spriteTexture = NULL;
    }

    if (renderer)
    {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }

    if (window)
    {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    SDL_Quit();
}
