import ctypes
import os

os.environ["SDL_MAIN_USE_CALLBACKS"] = "1"
os.environ["SDL_RENDER_DRIVER"] = "opengl"

import sdl3

renderer = ctypes.POINTER(sdl3.SDL_Renderer)()
window = ctypes.POINTER(sdl3.SDL_Window)()

mixer = ctypes.POINTER(sdl3.MIX_Mixer)()
audio = ctypes.POINTER(sdl3.MIX_Audio)()

@sdl3.SDL_AppInit_func
def SDL_AppInit(appstate, argc, argv):
    global mixer, audio

    sdl3.SDL_SetHint(b"SDL_MIXER_DISABLE_DRFLAC", b"1")
    sdl3.SDL_SetHint(b"SDL_MIXER_DISABLE_DRMP3", b"1")

    if not sdl3.SDL_Init(sdl3.SDL_INIT_VIDEO): # | sdl3.SDL_INIT_AUDIO
        sdl3.SDL_Log("Couldn't initialize SDL: %s".encode() % sdl3.SDL_GetError())
        return sdl3.SDL_APP_FAILURE

    title = "Play a sound by click using PySDL3".encode()
    if not sdl3.SDL_CreateWindowAndRenderer(title, 380, 380, 0, window, renderer):
        sdl3.SDL_Log("Couldn't create window/renderer: %s".encode() % sdl3.SDL_GetError())
        return sdl3.SDL_APP_FAILURE

    sdl3.SDL_SetRenderVSync(renderer, 1) # Turn on vertical sync

    # Get the compiled SDL version
    compiled = sdl3.SDL_VERSION
    compiled_major = sdl3.SDL_VERSIONNUM_MAJOR(compiled)
    compiled_minor = sdl3.SDL_VERSIONNUM_MINOR(compiled)
    compiled_micro = sdl3.SDL_VERSIONNUM_MICRO(compiled)
    print(f"Compiled SDL version: {compiled_major}.{compiled_minor}.{compiled_micro}")

    # Get the linked SDL version
    linked = sdl3.SDL_GetVersion()
    linked_major = sdl3.SDL_VERSIONNUM_MAJOR(linked)
    linked_minor = sdl3.SDL_VERSIONNUM_MINOR(linked)
    linked_micro = sdl3.SDL_VERSIONNUM_MICRO(linked)
    print(f"Linked SDL version:   {linked_major}.{linked_minor}.{linked_micro}")

    # v = sdl3.MIX_Version()
    # print(v)  # Output: 3001000
    # major = v // 1_000_000
    # minor = (v // 1_000) % 1000
    # micro = v % 1000
    # msg = f"SDL3_mixer version {major}.{minor}.{micro}".encode("utf-8")
    # sdl3.SDL_Log(msg) # Output: SDL3_mixer version 3.1.0

    v = sdl3.MIX_Version()
    major = sdl3.SDL_VERSIONNUM_MAJOR(v)
    minor = sdl3.SDL_VERSIONNUM_MINOR(v)
    micro = sdl3.SDL_VERSIONNUM_MICRO(v)
    msg = f"SDL3_mixer version:   {major}.{minor}.{micro}".encode("utf-8")
    sdl3.SDL_Log(msg)

    if not sdl3.MIX_Init():
        sdl3.SDL_Log("MIX_Init failed: %s".encode() % sdl3.SDL_GetError())
        return sdl3.SDL_APP_FAILURE

    mixer = sdl3.MIX_CreateMixerDevice(sdl3.SDL_AUDIO_DEVICE_DEFAULT_PLAYBACK, None)
    if not mixer:
        err = sdl3.SDL_GetError().decode("utf-8")
        sdl3.SDL_Log(f"Couldn't create mixer: {err}".encode())
        return sdl3.SDL_APP_FAILURE

    mixerspec = sdl3.SDL_AudioSpec()
    sdl3.MIX_GetMixerFormat(mixer, ctypes.byref(mixerspec))
    fmt_name = sdl3.SDL_GetAudioFormatName(mixerspec.format).decode('utf-8')
    msg = f"Mixer is format {fmt_name}, {mixerspec.channels} channels, {mixerspec.freq} frequency".encode()
    sdl3.SDL_Log(msg)

    # --- Print available audio decoders ---
    sdl3.SDL_Log(b"Available MIXER decoders:")

    num_decoders = sdl3.MIX_GetNumAudioDecoders()
    if num_decoders < 0:
        err = sdl3.SDL_GetError().decode("utf-8")
        msg = f" - [error ({err})]".encode("utf-8")
        sdl3.SDL_Log(msg)
    elif num_decoders == 0:
        sdl3.SDL_Log(b" - [none]")
    else:
        for i in range(num_decoders):
            decoder_name = sdl3.MIX_GetAudioDecoder(i).decode("utf-8")
            msg = f" - {decoder_name}".encode("utf-8")
            sdl3.SDL_Log(msg)

    # --- Load audio file ---
    audiofname = b"./assets/audio/picked-coin-echo-2.wav"  # bytes for SDL
    audio = sdl3.MIX_LoadAudio(mixer, audiofname, False)
    if not audio:
        err = sdl3.SDL_GetError().decode("utf-8")
        msg = f"Failed to load '{audiofname.decode()}' ({err})".encode("utf-8")
        sdl3.SDL_Log(msg)
    else:
        audiospec = sdl3.SDL_AudioSpec()
        sdl3.MIX_GetAudioFormat(audio, ctypes.byref(audiospec))

        fmt_name = sdl3.SDL_GetAudioFormatName(audiospec.format).decode("utf-8")
        channels_text = "" if audiospec.channels == 1 else "s"
        msg = f"{audiofname.decode()}: {fmt_name}, {audiospec.channels} channel{channels_text}, {audiospec.freq} freq".encode("utf-8")
        sdl3.SDL_Log(msg)

    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppEvent_func
def SDL_AppEvent(appstate, event):
    if sdl3.SDL_DEREFERENCE(event).type == sdl3.SDL_EVENT_QUIT:
        return sdl3.SDL_APP_SUCCESS
    elif sdl3.SDL_DEREFERENCE(event).type == sdl3.SDL_EVENT_MOUSE_BUTTON_DOWN:
        # Play the sound effect
        if not sdl3.MIX_PlayAudio(mixer, audio):
            sdl3.SDL_Log("Failed to play audio: %s".encode() % sdl3.SDL_GetError())

    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppIterate_func
def SDL_AppIterate(appstate):
    sdl3.SDL_SetRenderDrawColor(renderer, 210, 220, 210, sdl3.SDL_ALPHA_OPAQUE)
    sdl3.SDL_RenderClear(renderer)
    sdl3.SDL_RenderPresent(renderer)
    return sdl3.SDL_APP_CONTINUE

@sdl3.SDL_AppQuit_func
def SDL_AppQuit(appstate, result):
    global audio, mixer, renderer, window

    if audio:
        sdl3.MIX_DestroyAudio(audio)
        audio = None
    if mixer:
        sdl3.MIX_DestroyMixer(mixer)
        mixer = None

    if renderer:
        sdl3.SDL_DestroyRenderer(renderer)
        renderer = None
    if window:
        sdl3.SDL_DestroyWindow(window)
        window = None

    sdl3.MIX_Quit()
    sdl3.SDL_Quit()
